# Jarvis-Cognitive/ui/connector.py

import os
import subprocess
import sys
import requests
import streamlit as st

# --- Configurazione API ---
# L'URL del nostro backend Jarvis.
# Può essere sovrascritto via env: JARVIS_API_BASE_URL o API_BASE_URL
API_BASE_URL = os.getenv("JARVIS_API_BASE_URL", os.getenv("API_BASE_URL", "http://127.0.0.1:8000"))


def ask_jarvis_api(user_input: str, session_id: str = "default") -> str:
    """
    Invia una query all'API del backend Jarvis e restituisce la risposta.
    """
    endpoint = f"{API_BASE_URL}/ask"
    payload = {
        "query": user_input,
        "session_id": session_id
    }
    
    try:
        # Eseguiamo la richiesta POST all'API con un timeout generoso
        response = requests.post(endpoint, json=payload, timeout=120)
        
        # Solleva un'eccezione per codici di errore HTTP (4xx o 5xx)
        response.raise_for_status()
        
        # Estraiamo la risposta JSON
        response_data = response.json()
        
        print(f"[UI-CONNECTOR] Risposta API ricevuta: {response_data}")
        return response_data.get("answer", "La risposta dell'API non conteneva un campo 'answer'.")

    except requests.exceptions.ConnectionError:
        error_msg = "Errore di Connessione: Impossibile raggiungere il backend di Jarvis. Assicurati che il server API sia in esecuzione."
        st.error(error_msg)
        return error_msg
    except requests.exceptions.Timeout:
        error_msg = "Timeout: Il backend non ha risposto in tempo. Potrebbe essere sovraccarico o impegnato in un'operazione lunga."
        st.error(error_msg)
        return error_msg
    except requests.exceptions.RequestException as e:
        error_msg = f"Errore API: {e}"
        st.error(error_msg)
        return error_msg
    except Exception as e:
        error_msg = f"Errore imprevisto nel connettore: {e}"
        st.error(error_msg)
        return error_msg

# La funzione per l'upload dei file rimane per ora, anche se la sua logica
# andrà rivista in futuro per usare un endpoint API dedicato.
ROOT_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))

def gestisci_upload_e_indicizzazione(uploaded_file, save_source: bool = True):
    """
    Salva temporaneamente il file caricato e lancia lo script di indicizzazione
    avanzata in un processo separato.
    """
    temp_dir = os.path.join(ROOT_DIR, "data", "temp_uploads")
    os.makedirs(temp_dir, exist_ok=True)
    file_path = os.path.join(temp_dir, uploaded_file.name)
    
    with open(file_path, "wb") as f:
        f.write(uploaded_file.getbuffer())
    
    st.info(f"File '{uploaded_file.name}' ricevuto. Avvio del processo di indicizzazione avanzata...")

    script_path = os.path.join(ROOT_DIR, "scripts", "indicizza_documenti.py")
    
    # Eseguiamo lo script di indicizzazione in un sottoprocesso
    # Questo evita di bloccare la UI di Streamlit
    # Determina profilo corrente dalla env
    profile = os.getenv('AGENT_PROFILE', 'aurelio')
    cmd = [sys.executable, script_path, file_path, '--profile', profile]
    if not save_source:
        cmd.append('--no-save-source')

    with st.spinner("Indicizzazione in corso... Questa operazione potrebbe richiedere alcuni minuti."):
        result = subprocess.run(cmd, capture_output=True, text=True)

    if result.returncode == 0:
        st.success("Indicizzazione completata con successo!")
        st.code(result.stdout)
    else:
        st.error("Si è verificato un errore durante l'indicizzazione:")
        st.code(result.stderr)
    
    # Rimuoviamo il file temporaneo
    os.remove(file_path)
    
