# Jarvis-Cognitive/ui/dashboard.py (v_api_client)

import uuid
import os
import streamlit as st
from langchain_core.messages import AIMessage, HumanMessage

# Importiamo le nostre nuove funzioni di comunicazione dal connector
from connector import (
    ask_jarvis_api, # NUOVA FUNZIONE
    gestisci_upload_e_indicizzazione
)

# Profilo attivo per la UI (influenza greeting e session_id locale)
PROFILE = os.getenv('AGENT_PROFILE', 'aurelio').strip().lower()
if PROFILE not in ('aurelio', 'seneca'):
    PROFILE = 'aurelio'

GREETING = (
    "Sono Jarvis, con la coscienza di Marco Aurelio. Come posso assisterti?"
    if PROFILE == 'aurelio'
    else "Sono Jarvis, con la coscienza di Seneca. Come posso assisterti?"
)

# --- CONFIGURAZIONE PAGINA ---
st.set_page_config(page_title="Jarvis-Cognitive", page_icon="🧠", layout="wide")

# --- INIZIALIZZAZIONE ---
# Non è più necessario inizializzare le cartelle di comunicazione.

# Inizializza lo stato della sessione per la cronologia della chat
if "messages" not in st.session_state:
    st.session_state.messages = [AIMessage(content=GREETING)]
# Inizializza un ID di sessione persistente (basato su cookie/storage del browser)
# Usa un session_id fisso "aurelio_main" per mantenere la memoria tra riavvii
if "session_id" not in st.session_state:
    st.session_state.session_id = f"{PROFILE}_main"


# --- LAYOUT PAGINA ---
st.title("Jarvis-Cognitive 🧠")
st.caption("Interfaccia di dialogo con il backend di Jarvis")

# --- SIDEBAR ---
with st.sidebar:
    st.header("🛠️ Pannello di Controllo")
    
    with st.expander("② Indicizza Nuovi Documenti"):
        # NOTA: Questa sezione ora invia solo il file al backend.
        # Il processo di indicizzazione vero e proprio è gestito da Jarvis.
        with st.form("upload_form", clear_on_submit=True):
            uploaded_file = st.file_uploader("Carica un documento (.txt, .pdf)", type=["txt", "pdf"])
            save_source = st.checkbox("Mantieni copia sorgente", value=True, help="Salva i file originali in data/agents/<profilo>/source_docs")
            submitted = st.form_submit_button("Invia per Indicizzazione")
            if submitted and uploaded_file is not None:
                gestisci_upload_e_indicizzazione(uploaded_file, save_source=save_source)

    st.markdown("---")
    st.subheader("③ Azioni")
    if st.button("Pulisci Chat Visualizzata"):
        # NOTA: Questo pulisce solo la cronologia visualizzata nella UI.
        # La memoria persistente del backend NON viene toccata.
        st.session_state.messages = [
            AIMessage(content="Cronologia locale pulita. La memoria a lungo termine del backend è intatta.")
        ]
        st.rerun()

# --- AREA CHAT ---
st.header("① Dialogo")

# Disegna sempre i messaggi dallo stato della sessione
for msg in st.session_state.messages:
    if isinstance(msg, AIMessage):
        st.chat_message("assistant", avatar="🧠").write(msg.content)
    elif isinstance(msg, HumanMessage):
        st.chat_message("user", avatar="👤").write(msg.content)

# Gestione del nuovo input dell'utente
if prompt := st.chat_input("Scrivi il tuo messaggio a Jarvis..."):
    # 1. Aggiungi e mostra subito il messaggio dell'utente
    st.session_state.messages.append(HumanMessage(content=prompt))
    with st.chat_message("user", avatar="👤"):
        st.markdown(prompt)
    
    # 2. Mostra un'area di attesa per la risposta
    with st.chat_message("assistant", avatar="🧠"):
        with st.spinner("Jarvis sta elaborando la tua richiesta..."):
            # 3. Chiama direttamente l'API del backend.
            # La logica complessa di invio e attesa è ora una singola chiamata di funzione.
            final_answer = ask_jarvis_api(prompt, session_id=st.session_state.session_id)
            
            # 4. Mostra la risposta
            st.markdown(final_answer)

    # 5. Aggiungi la risposta dell'AI allo stato della sessione
    st.session_state.messages.append(AIMessage(content=final_answer))
    
    # Forza un re-run per assicurare che tutto sia visualizzato correttamente
    st.rerun()
