# Jarvis-Cognitive/services/cognitiveservice/cognitiveservice.py (v_final_db_fix)

import os
import re
import time
import sqlite3 # Importa la libreria nativa di Python per SQLite
from dotenv import load_dotenv

from langchain_google_genai import ChatGoogleGenerativeAI
from langchain_community.chat_message_histories import SQLChatMessageHistory
from core.cognitive_engine.agent import AgenteFinale 

class CognitiveService:
    """
    Servizio Jarvis che funge da ponte tra il Kernel e il motore cognitivo (Agente).
    (v_final: Include debouncing e fix per il percorso del DB della memoria)
    """
    def __init__(self, core_api, config):
        self.core = core_api
        self.config = config
        self.agent_executor = None
        self.history_handler = None
        self.recently_processed = {}
        
        self.log("Inizializzazione servizio cognitivo...")

        env_path = os.path.join(self.core.base_dir, 'data', '.env')
        load_dotenv(dotenv_path=env_path)

        llm_config = self.config.get('llm', {})
        conversation_db_path = self.config.get('conversation_db_path')
        system_prompt = self.config.get('system_prompt', "Sei un assistente AI generico.")

        if not conversation_db_path:
            self.log("Configurazione incompleta (manca 'conversation_db_path'). Servizio disabilitato.", 'error')
            return

        # --- GESTIONE ESPLICITA DEL DATABASE DELLA MEMORIA ---
        # Costruisce il percorso assoluto e corretto per il database
        abs_conv_db_path = os.path.join(self.core.base_dir, conversation_db_path)
        
        try:
            self.log(f"Verifica/Creazione del database della memoria in: {abs_conv_db_path}")
            # Assicura che la directory genitore esista
            os.makedirs(os.path.dirname(abs_conv_db_path), exist_ok=True)
            # Connessione (crea il file .sqlite se non esiste)
            con = sqlite3.connect(abs_conv_db_path)
            cur = con.cursor()
            # Crea la tabella 'message_store' se non esiste, come richiesto da LangChain
            cur.execute("""
                CREATE TABLE IF NOT EXISTS message_store (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    session_id TEXT NOT NULL,
                    message TEXT NOT NULL
                )
            """)
            con.commit()
            con.close()
            self.log("Database della memoria pronto.")
            # --- FINE GESTIONE ESPLICITA ---

            llm = ChatGoogleGenerativeAI(
                model=llm_config.get('model', 'gemini-1.5-pro-latest'),
                temperature=llm_config.get('temperature', 0.2)
            )
            
            self.agente = AgenteFinale(llm=llm, system_prompt=system_prompt)
            self.agent_executor = self.agente.agent_executor

            self.history_handler = SQLChatMessageHistory(
                session_id="sessione_principale",
                connection_string=f"sqlite:///{abs_conv_db_path}"
            )
            
            self.log("Servizio cognitivo inizializzato con successo. Agente pronto.")

        except Exception as e:
            self.log(f"Errore critico durante l'inizializzazione: {e}", "error")
            self.agent_executor = None

    def start(self):
        if not self.agent_executor:
            self.log("Servizio non avviato a causa di un errore di inizializzazione.", "warning")
            return
            
        self.log("Servizio avviato. In ascolto per eventi 'ui:new_query_file'.")
        self.core.subscribe_to_event('ui:new_query_file', self.handle_new_query_from_ui)

    def stop(self):
        self.log("Servizio fermato.")

    def log(self, message, level='info'):
        self.core.log(f"[COGNITIVE] {message}", level)

    def handle_new_query_from_ui(self, data):
        query_filepath = data.get('path')
        current_time = time.time()
        
        if query_filepath in self.recently_processed and (current_time - self.recently_processed.get(query_filepath, 0)) < 2:
            self.log(f"Evento duplicato ignorato per: {os.path.basename(query_filepath)}", 'debug')
            return
        
        self.recently_processed[query_filepath] = current_time
        self.recently_processed = {k: v for k, v in self.recently_processed.items() if current_time - v < 60}
        
        if not self.agent_executor:
            self.log("Richiesta ignorata, servizio non operativo.", "warning")
            return

        if not query_filepath or not os.path.exists(query_filepath):
            self.log(f"Evento ricevuto ma il file non esiste: {query_filepath}", 'warning')
            return
        
        request_id = None
        match = re.search(r'query_([a-f0-9-]+)\.txt', os.path.basename(query_filepath))
        if match:
            request_id = match.group(1)
        else:
            self.log(f"Nome del file di query non valido: {os.path.basename(query_filepath)}", "warning")
            os.remove(query_filepath)
            return

        try:
            with open(query_filepath, 'r', encoding='utf-8') as f:
                user_query = f.read().strip()

            if not user_query:
                self.log(f"File di query '{os.path.basename(query_filepath)}' vuoto.", "warning")
                self._write_response_file(request_id, "")
                return

            self.log(f"Nuova query [ID: {request_id}]: '{user_query[:50]}...'")

            chat_history = self.history_handler.messages
            
            response_dict = self.agent_executor.invoke({
                "input": user_query,
                "chat_history": chat_history
            })
            
            final_answer = response_dict.get("output", "Errore: l'agente non ha fornito una risposta.")

            self.history_handler.add_user_message(user_query)
            self.history_handler.add_ai_message(final_answer)

            self.log(f"Risposta generata [ID: {request_id}]: '{final_answer[:50]}...'")
            self._write_response_file(request_id, final_answer)

        except Exception as e:
            error_message = f"Errore imprevisto durante l'elaborazione della query: {e}"
            self.log(error_message, "error")
            self._write_response_file(request_id, error_message)
        finally:
            if os.path.exists(query_filepath):
                os.remove(query_filepath)
    
    def _write_response_file(self, request_id: str, content: str):
        try:
            output_dir = os.path.join(self.core.base_dir, "data", "comms", "output")
            os.makedirs(output_dir, exist_ok=True)
            response_filename = f"response_{request_id}.txt"
            response_filepath = os.path.join(output_dir, response_filename)
            
            with open(response_filepath, 'w', encoding='utf-8') as f:
                f.write(content)
            self.log(f"File di risposta scritto per ID: {request_id}", 'debug')
        except Exception as e:
            self.log(f"Impossibile scrivere il file di risposta per ID {request_id}: {e}", "error")