# services/fileloggerservice.py
import os
from datetime import datetime

class FileLoggerService:
    """
    Servizio di backend per la scrittura persistente dei log su file.
    È completamente headless.
    """
    def __init__(self, core_api, config):
        self.core = core_api
        
        # Legge la directory dei log dalla configurazione
        log_dir = config.get('log_directory', './logs') # Aggiunto fallback a una cartella locale
        
        # Espande eventuali percorsi relativi o con '~'
        log_dir = os.path.expanduser(log_dir)
        if not os.path.isabs(log_dir):
            log_dir = os.path.join(self.core.base_dir, log_dir)
            
        # --- MODIFICA CHIAVE ---
        # Crea la directory dei log se non esiste.
        # `exist_ok=True` previene errori se la cartella è già presente.
        try:
            os.makedirs(log_dir, exist_ok=True)
            self.log_file_path = os.path.join(log_dir, "jarvis.log")
        except OSError as e:
            # Se c'è un errore nella creazione della cartella (es. permessi)
            # il logger non può funzionare. Stampiamo un errore critico.
            print(f"[FILELOGGER] ERRORE CRITICO: Impossibile creare la directory di log {log_dir}: {e}")
            self.log_file_path = None # Disabilita il logger

        # Si iscrive all'evento di log del Core
        self.core.subscribe_to_event("log:new_message", self.write_log)

    def start(self):
        if self.log_file_path:
            self.log(f"Servizio avviato. I log verranno scritti su: {self.log_file_path}", 'info')

    def stop(self):
        if self.log_file_path:
            self.log("Servizio fermato.", 'info')
    
    def log(self, message, level='info'):
        """Metodo di logging per il servizio stesso."""
        # Aggiunge un prefisso per identificare i log di questo servizio
        self.core.log(f"[FILELOGGER] {message}", level)

    def write_log(self, log_data):
        """
        Callback eseguito per ogni evento 'log:new_message'.
        Scrive il messaggio ricevuto nel file di log.
        """
        if not self.log_file_path:
            return # Il logger è disabilitato

        message = log_data.get('message', '')
        level = log_data.get('level', 'info').upper()
        timestamp = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        
        log_line = f"{timestamp} [{level}] - {message}\n"
        
        try:
            with open(self.log_file_path, 'a', encoding='utf-8') as f:
                f.write(log_line)
        except Exception as e:
            # Se il logger fallisce, l'unico modo per segnalarlo è stampare sulla console
            print(f"ERRORE CRITICO NEL FILELOGGER: Impossibile scrivere su file di log! {e}")