# Jarvis-Cognitive/ui/connector.py

import os
import time
import uuid
import streamlit as st

# --- Percorsi di Comunicazione ---
# Questi percorsi sono relativi alla posizione dello script dashboard.py
# quindi dobbiamo risalire di un livello per trovare la root del progetto.
ROOT_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
INPUT_DIR = os.path.join(ROOT_DIR, "data", "comms", "input")
OUTPUT_DIR = os.path.join(ROOT_DIR, "data", "comms", "output")


def initialize_communication_dirs():
    """Assicura che le cartelle di comunicazione esistano."""
    os.makedirs(INPUT_DIR, exist_ok=True)
    os.makedirs(OUTPUT_DIR, exist_ok=True)


def send_query_to_backend(user_input: str) -> str:
    """
    Invia una query al backend Jarvis scrivendola su un file.
    Restituisce l'ID univoco della richiesta.
    """
    request_id = str(uuid.uuid4())
    query_filename = f"query_{request_id}.txt"
    query_filepath = os.path.join(INPUT_DIR, query_filename)

    try:
        with open(query_filepath, 'w', encoding='utf-8') as f:
            f.write(user_input)
        
        print(f"[UI-CONNECTOR] Query inviata con ID: {request_id}")
        return request_id
    except Exception as e:
        st.error(f"Errore nell'invio della richiesta al backend: {e}")
        return None


def wait_for_backend_response(request_id: str, timeout_seconds: int = 60) -> str:
    """
    Attende la risposta dal backend controllando la cartella di output.
    Restituisce il contenuto della risposta o un messaggio di errore.
    """
    response_filename = f"response_{request_id}.txt"
    response_filepath = os.path.join(OUTPUT_DIR, response_filename)
    
    start_time = time.time()
    
    while time.time() - start_time < timeout_seconds:
        if os.path.exists(response_filepath):
            try:
                with open(response_filepath, 'r', encoding='utf-8') as f:
                    response_content = f.read()
                
                # Pulisce il file di risposta dopo averlo letto
                os.remove(response_filepath)
                print(f"[UI-CONNECTOR] Risposta ricevuta per ID: {request_id}")
                return response_content
            except Exception as e:
                return f"Errore nella lettura della risposta dal backend: {e}"
        
        time.sleep(0.5) # Attende mezzo secondo tra un controllo e l'altro

    return "Timeout: Il backend non ha risposto in tempo."

# --- Funzioni del vecchio connector da adattare o rimuovere ---

def gestisci_upload_e_indicizzazione(uploaded_file):
    """
    Adatta la vecchia funzione per salvare i file in una cartella che
    un altro servizio Jarvis (es. FileSystemWatcher) può monitorare.
    """
    # Per ora, semplicemente salviamo il file in una cartella di "pending"
    # L'indicizzazione vera e propria sarà un'azione del backend
    pending_dir = os.path.join(ROOT_DIR, "data", "documenti_da_indicizzare")
    os.makedirs(pending_dir, exist_ok=True)
    file_path = os.path.join(pending_dir, uploaded_file.name)
    
    with open(file_path, "wb") as f:
        f.write(uploaded_file.getbuffer())
    
    st.success(f"File '{uploaded_file.name}' inviato al backend per l'indicizzazione.")
    # In una versione futura, potremmo attendere una conferma dal backend
    