# services/fileloggerservice/fileloggerservice.py

import os
from datetime import datetime

class FileLoggerService:
    """
    Servizio di backend per la scrittura persistente dei log su file.
    """
    def __init__(self, core_api, config):
        self.core = core_api
        
        # --- MODIFICA CHIAVE: Utilizzo dell'API del Kernel ---
        # La configurazione ora contiene solo il percorso *relativo* a /data
        log_path_from_config = config.get('log_file_path', 'logs/jarvis.log')
        
        try:
            # Il Kernel gestisce la creazione della cartella 'data/logs' e restituisce il percorso completo.
            self.log_file_path = self.core.get_data_path(log_path_from_config)
        except OSError as e:
            print(f"[FILELOGGER] ERRORE CRITICO: Impossibile creare percorso di log via Kernel: {e}")
            self.log_file_path = None # Disabilita il logger
        # --- FINE MODIFICA ---

        self.core.subscribe_to_event("log:new_message", self.write_log)

    def start(self):
        if self.log_file_path:
            self.log(f"Servizio avviato. I log verranno scritti su: {self.log_file_path}", 'info')

    def stop(self):
        if self.log_file_path:
            self.log("Servizio fermato.", 'info')
    
    def log(self, message, level='info'):
        self.core.log(f"[FILELOGGER] {message}", level)

    def write_log(self, log_data):
        if not self.log_file_path:
            return

        message = log_data.get('message', '')
        level = log_data.get('level', 'info').upper()
        timestamp = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        
        log_line = f"{timestamp} [{level}] - {message}\n"
        
        try:
            with open(self.log_file_path, 'a', encoding='utf-8') as f:
                f.write(log_line)
        except Exception as e:
            print(f"ERRORE CRITICO NEL FILELOGGER: Impossibile scrivere su file di log! {e}")