# Jarvis-Cognitive/ui/dashboard.py (v_cognitive_backend)

import streamlit as st
from langchain_core.messages import AIMessage, HumanMessage

# Importiamo le nostre nuove funzioni di comunicazione dal connector
from connector import (
    initialize_communication_dirs, 
    send_query_to_backend, 
    wait_for_backend_response,
    gestisci_upload_e_indicizzazione
)

# --- CONFIGURAZIONE PAGINA ---
st.set_page_config(page_title="Jarvis-Cognitive", page_icon="🧠", layout="wide")

# --- INIZIALIZZAZIONE ---
# Assicura che le cartelle per la comunicazione esistano
initialize_communication_dirs()

# Inizializza lo stato della sessione per la cronologia della chat
if "messages" not in st.session_state:
    st.session_state.messages = [
        AIMessage(content="Sono Jarvis, con la coscienza di Marco Aurelio. Come posso assisterti?")
    ]

# --- LAYOUT PAGINA ---
st.title("Jarvis-Cognitive 🧠")
st.caption("Interfaccia di dialogo con il backend di Jarvis")

# --- SIDEBAR ---
with st.sidebar:
    st.header("🛠️ Pannello di Controllo")
    
    with st.expander("② Indicizza Nuovi Documenti"):
        # NOTA: Questa sezione ora invia solo il file al backend.
        # Il processo di indicizzazione vero e proprio è gestito da Jarvis.
        with st.form("upload_form", clear_on_submit=True):
            uploaded_file = st.file_uploader("Carica un documento (.txt, .pdf)", type=["txt", "pdf"])
            submitted = st.form_submit_button("Invia per Indicizzazione")
            if submitted and uploaded_file is not None:
                gestisci_upload_e_indicizzazione(uploaded_file)

    st.markdown("---")
    st.subheader("③ Azioni")
    if st.button("Pulisci Chat Visualizzata"):
        # NOTA: Questo pulisce solo la cronologia visualizzata nella UI.
        # La memoria persistente del backend NON viene toccata.
        st.session_state.messages = [
            AIMessage(content="Cronologia locale pulita. La memoria a lungo termine del backend è intatta.")
        ]
        st.rerun()

# --- AREA CHAT ---
st.header("① Dialogo")

# Disegna sempre i messaggi dallo stato della sessione
for msg in st.session_state.messages:
    if isinstance(msg, AIMessage):
        st.chat_message("assistant", avatar="🧠").write(msg.content)
    elif isinstance(msg, HumanMessage):
        st.chat_message("user", avatar="👤").write(msg.content)

# Gestione del nuovo input dell'utente
if prompt := st.chat_input("Scrivi il tuo messaggio a Jarvis..."):
    # 1. Aggiungi e mostra subito il messaggio dell'utente
    st.session_state.messages.append(HumanMessage(content=prompt))
    with st.chat_message("user", avatar="👤"):
        st.markdown(prompt)
    
    # 2. Mostra un'area di attesa per la risposta
    with st.chat_message("assistant", avatar="🧠"):
        with st.spinner("Invio richiesta al backend di Jarvis... In attesa di risposta..."):
            # 3. Invia la query al backend e ottieni un ID di richiesta
            request_id = send_query_to_backend(prompt)
            
            if request_id:
                # 4. Attendi la risposta dal backend
                final_answer = wait_for_backend_response(request_id)
            else:
                final_answer = "Errore: Impossibile inviare la richiesta al backend."

            # 5. Mostra la risposta
            st.markdown(final_answer)

    # 6. Aggiungi la risposta dell'AI allo stato della sessione
    st.session_state.messages.append(AIMessage(content=final_answer))
    
    # Forza un re-run per assicurare che tutto sia visualizzato correttamente
    # anche se non strettamente necessario con il nuovo approccio di Streamlit
    st.rerun()